<?php
/**
 * Authentifizierung: LDAP + lokaler Admin Fallback
 */

require_once __DIR__ . '/config.php';

/**
 * Authentifiziert einen Benutzer
 * Prüft erst lokalen Admin, dann LDAP
 */
function authenticate(string $username, string $password): array|false {
    // Leere Eingaben ablehnen
    if (empty($username) || empty($password)) {
        return false;
    }

    // Zuerst lokalen Admin prüfen (immer verfügbar als Fallback)
    if (LOCAL_ADMIN_ENABLED && $username === LOCAL_ADMIN_USER) {
        if (password_verify($password, LOCAL_ADMIN_PASS_HASH)) {
            logAuth($username, 'local', true);
            return ['method' => 'local', 'username' => $username];
        }
    }

    // Dann LDAP versuchen (wenn aktiviert)
    if (LDAP_ENABLED) {
        $ldap_result = ldap_authenticate($username, $password);
        if ($ldap_result) {
            logAuth($username, 'ldap', true);
            return ['method' => 'ldap', 'username' => $username];
        }
    }

    logAuth($username, LDAP_ENABLED ? 'ldap/local' : 'local', false);
    return false;
}

/**
 * LDAP-Authentifizierung
 */
function ldap_authenticate(string $username, string $password): bool {
    // Prüfen ob LDAP-Extension verfügbar
    if (!function_exists('ldap_connect')) {
        error_log('LDAP: Extension nicht verfügbar');
        return false;
    }

    // Verbindung zum LDAP-Server
    $ldap_conn = @ldap_connect(LDAP_SERVER, LDAP_PORT);
    if (!$ldap_conn) {
        error_log('LDAP: Verbindung fehlgeschlagen zu ' . LDAP_SERVER);
        return false;
    }

    // LDAP-Optionen setzen
    ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3);
    ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0);
    ldap_set_option($ldap_conn, LDAP_OPT_NETWORK_TIMEOUT, 5);

    // DN konstruieren (Username einfügen)
    $bind_dn = str_replace('%username%', ldap_escape($username, '', LDAP_ESCAPE_DN), LDAP_BIND_DN);

    // Authentifizierung versuchen
    $bind = @ldap_bind($ldap_conn, $bind_dn, $password);

    if ($bind) {
        // Optional: Gruppen-Zugehörigkeit prüfen
        if (LDAP_USE_GROUP_CHECK) {
            $is_admin = check_ldap_group($ldap_conn, $username);
            ldap_close($ldap_conn);
            return $is_admin;
        }

        ldap_close($ldap_conn);
        return true;
    } else {
        $error = ldap_error($ldap_conn);
        error_log("LDAP: Auth fehlgeschlagen für User: $username - $error");
        ldap_close($ldap_conn);
        return false;
    }
}

/**
 * Prüft Gruppen-Zugehörigkeit im LDAP
 */
function check_ldap_group($ldap_conn, string $username): bool {
    $search = @ldap_search(
        $ldap_conn,
        LDAP_ADMIN_GROUP,
        "(memberUid=" . ldap_escape($username, '', LDAP_ESCAPE_FILTER) . ")"
    );

    if (!$search) {
        error_log("LDAP: Gruppensuche fehlgeschlagen");
        return false;
    }

    $entries = ldap_get_entries($ldap_conn, $search);
    return ($entries['count'] > 0);
}

/**
 * Prüft ob Benutzer eingeloggt ist
 */
function is_logged_in(): bool {
    return isset($_SESSION['authenticated']) && $_SESSION['authenticated'] === true;
}

/**
 * Erfordert Login, leitet sonst zur Login-Seite weiter
 */
function require_login(): void {
    init_session();

    if (!is_logged_in()) {
        header('Location: login.php');
        exit;
    }

    // Session-Timeout prüfen
    if (isset($_SESSION['login_time']) && (time() - $_SESSION['login_time'] > SESSION_TIMEOUT)) {
        session_destroy();
        session_start();
        $_SESSION['flash_message'] = 'Ihre Sitzung ist abgelaufen. Bitte melden Sie sich erneut an.';
        header('Location: login.php');
        exit;
    }

    // Session-Zeit aktualisieren
    $_SESSION['login_time'] = time();
}

/**
 * Benutzer ausloggen
 */
function logout(): void {
    init_session();

    $username = $_SESSION['username'] ?? 'unknown';
    logAuth($username, 'logout', true);

    $_SESSION = [];

    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }

    session_destroy();
}

/**
 * Gibt den aktuellen Benutzernamen zurück
 */
function get_logged_in_user(): ?string {
    return $_SESSION['username'] ?? null;
}

/**
 * Gibt die Login-Methode zurück
 */
function get_auth_method(): ?string {
    return $_SESSION['auth_method'] ?? null;
}

/**
 * Logging für Auth-Vorgänge
 */
function logAuth(string $username, string $method, bool $success): void {
    $status = $success ? 'SUCCESS' : 'FAILED';
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $message = date('Y-m-d H:i:s') . " [$status] User: $username, Method: $method, IP: $ip\n";

    $logFile = LOG_PATH . 'auth.log';
    @file_put_contents($logFile, $message, FILE_APPEND | LOCK_EX);
}
