<?php
/**
 * Datenbank-Verbindung und Query-Funktionen
 */

require_once __DIR__ . '/config.php';

class Database {
    private static ?PDO $instance = null;

    public static function getInstance(): PDO {
        if (self::$instance === null) {
            try {
                if (DB_TYPE === 'pgsql') {
                    $dsn = "pgsql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME;
                } else {
                    $dsn = "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=utf8mb4";
                }

                self::$instance = new PDO($dsn, DB_USER, DB_PASS, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]);
            } catch (PDOException $e) {
                error_log("Datenbankverbindung fehlgeschlagen: " . $e->getMessage());
                throw new Exception("Datenbankverbindung konnte nicht hergestellt werden.");
            }
        }
        return self::$instance;
    }

    // Verbindung testen
    public static function testConnection(): bool {
        try {
            $pdo = self::getInstance();
            $pdo->query("SELECT 1");
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}

/**
 * Bestellungen-Funktionen
 */

function createBestellung(array $data): int {
    $pdo = Database::getInstance();

    $sql = "INSERT INTO bestellungen (
        schul_name, software_name, version, hersteller, hersteller_website, isbn,
        software_quelle, download_pfad, lizenz_typ, lizenzschluessel, lizenzserver_anmerkungen
    ) VALUES (
        :schul_name, :software_name, :version, :hersteller, :hersteller_website, :isbn,
        :software_quelle, :download_pfad, :lizenz_typ, :lizenzschluessel, :lizenzserver_anmerkungen
    )";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'schul_name' => $data['schul_name'],
        'software_name' => $data['software_name'],
        'version' => $data['version'],
        'hersteller' => $data['hersteller'],
        'hersteller_website' => $data['hersteller_website'],
        'isbn' => $data['isbn'] ?? null,
        'software_quelle' => $data['software_quelle'],
        'download_pfad' => $data['download_pfad'] ?? null,
        'lizenz_typ' => $data['lizenz_typ'],
        'lizenzschluessel' => $data['lizenzschluessel'] ?? null,
        'lizenzserver_anmerkungen' => $data['lizenzserver_anmerkungen'] ?? null
    ]);

    return (int) $pdo->lastInsertId(DB_TYPE === 'pgsql' ? 'bestellungen_id_seq' : null);
}

function getBestellung(int $id): ?array {
    $pdo = Database::getInstance();
    $stmt = $pdo->prepare("SELECT * FROM bestellungen WHERE id = ?");
    $stmt->execute([$id]);
    $result = $stmt->fetch();
    return $result ?: null;
}

function getAllBestellungen(array $filters = [], int $page = 1, int $limit = ITEMS_PER_PAGE): array {
    $pdo = Database::getInstance();

    $where = [];
    $params = [];

    if (!empty($filters['status'])) {
        $where[] = "status = :status";
        $params['status'] = $filters['status'];
    }

    if (!empty($filters['schul_name'])) {
        $where[] = "schul_name ILIKE :schul_name";
        $params['schul_name'] = '%' . $filters['schul_name'] . '%';
    }

    if (!empty($filters['software_name'])) {
        $where[] = "software_name ILIKE :software_name";
        $params['software_name'] = '%' . $filters['software_name'] . '%';
    }

    $whereClause = count($where) > 0 ? 'WHERE ' . implode(' AND ', $where) : '';

    $offset = ($page - 1) * $limit;

    // MySQL verwendet LIKE statt ILIKE
    if (DB_TYPE === 'mysql') {
        $whereClause = str_replace('ILIKE', 'LIKE', $whereClause);
    }

    $sql = "SELECT * FROM bestellungen $whereClause ORDER BY erstellt_am DESC LIMIT $limit OFFSET $offset";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);

    return $stmt->fetchAll();
}

function countBestellungen(array $filters = []): int {
    $pdo = Database::getInstance();

    $where = [];
    $params = [];

    if (!empty($filters['status'])) {
        $where[] = "status = :status";
        $params['status'] = $filters['status'];
    }

    if (!empty($filters['schul_name'])) {
        $where[] = "schul_name ILIKE :schul_name";
        $params['schul_name'] = '%' . $filters['schul_name'] . '%';
    }

    if (!empty($filters['software_name'])) {
        $where[] = "software_name ILIKE :software_name";
        $params['software_name'] = '%' . $filters['software_name'] . '%';
    }

    $whereClause = count($where) > 0 ? 'WHERE ' . implode(' AND ', $where) : '';

    if (DB_TYPE === 'mysql') {
        $whereClause = str_replace('ILIKE', 'LIKE', $whereClause);
    }

    $sql = "SELECT COUNT(*) FROM bestellungen $whereClause";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);

    return (int) $stmt->fetchColumn();
}

function updateBestellung(int $id, array $data): bool {
    $pdo = Database::getInstance();

    // Hole alten Zustand für Historie
    $oldData = getBestellung($id);
    if (!$oldData) {
        return false;
    }

    $allowedFields = [
        'schul_name', 'software_name', 'version', 'hersteller', 'hersteller_website', 'isbn',
        'software_quelle', 'download_pfad', 'lizenz_typ', 'lizenzschluessel',
        'lizenzserver_anmerkungen', 'status', 'notizen'
    ];

    $updates = [];
    $params = ['id' => $id];

    foreach ($allowedFields as $field) {
        if (array_key_exists($field, $data)) {
            $updates[] = "$field = :$field";
            $params[$field] = $data[$field];

            // Historie-Eintrag erstellen wenn Wert sich geändert hat
            if ($oldData[$field] !== $data[$field]) {
                logHistorie($id, $field, $oldData[$field], $data[$field], $data['bearbeitet_von'] ?? null);
            }
        }
    }

    if (empty($updates)) {
        return true;
    }

    $updates[] = "bearbeitet_am = CURRENT_TIMESTAMP";
    if (!empty($data['bearbeitet_von'])) {
        $updates[] = "bearbeitet_von = :bearbeitet_von";
        $params['bearbeitet_von'] = $data['bearbeitet_von'];
    }

    $sql = "UPDATE bestellungen SET " . implode(', ', $updates) . " WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    return $stmt->execute($params);
}

function logHistorie(int $bestellungId, string $feld, ?string $alterWert, ?string $neuerWert, ?string $bearbeiter): void {
    $pdo = Database::getInstance();

    $sql = "INSERT INTO bestellungen_historie (bestellung_id, feld, alter_wert, neuer_wert, aenderung_von)
            VALUES (:bestellung_id, :feld, :alter_wert, :neuer_wert, :aenderung_von)";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'bestellung_id' => $bestellungId,
        'feld' => $feld,
        'alter_wert' => $alterWert,
        'neuer_wert' => $neuerWert,
        'aenderung_von' => $bearbeiter
    ]);
}

function getHistorie(int $bestellungId): array {
    $pdo = Database::getInstance();
    $stmt = $pdo->prepare("SELECT * FROM bestellungen_historie WHERE bestellung_id = ? ORDER BY aenderung_am DESC");
    $stmt->execute([$bestellungId]);
    return $stmt->fetchAll();
}

function getStatusCounts(): array {
    $pdo = Database::getInstance();
    $stmt = $pdo->query("SELECT status, COUNT(*) as count FROM bestellungen GROUP BY status");
    $result = [];
    while ($row = $stmt->fetch()) {
        $result[$row['status']] = (int) $row['count'];
    }
    return $result;
}

/**
 * Archiviert eine Bestellung (verschiebt sie in die Archiv-Tabelle)
 */
function archiveBestellung(int $id, string $archiviertVon): bool {
    $pdo = Database::getInstance();

    $bestellung = getBestellung($id);
    if (!$bestellung) {
        return false;
    }

    try {
        $pdo->beginTransaction();

        // In Archiv einfügen
        $sql = "INSERT INTO bestellungen_archiv (
            id, schul_name, software_name, version, hersteller, hersteller_website, isbn,
            software_quelle, download_pfad, lizenz_typ, lizenzschluessel, lizenzserver_anmerkungen,
            erstellt_am, bearbeitet_am, bearbeitet_von, notizen, archiviert_von
        ) VALUES (
            :id, :schul_name, :software_name, :version, :hersteller, :hersteller_website, :isbn,
            :software_quelle, :download_pfad, :lizenz_typ, :lizenzschluessel, :lizenzserver_anmerkungen,
            :erstellt_am, :bearbeitet_am, :bearbeitet_von, :notizen, :archiviert_von
        )";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'id' => $bestellung['id'],
            'schul_name' => $bestellung['schul_name'],
            'software_name' => $bestellung['software_name'],
            'version' => $bestellung['version'],
            'hersteller' => $bestellung['hersteller'],
            'hersteller_website' => $bestellung['hersteller_website'],
            'isbn' => $bestellung['isbn'],
            'software_quelle' => $bestellung['software_quelle'],
            'download_pfad' => $bestellung['download_pfad'],
            'lizenz_typ' => $bestellung['lizenz_typ'],
            'lizenzschluessel' => $bestellung['lizenzschluessel'],
            'lizenzserver_anmerkungen' => $bestellung['lizenzserver_anmerkungen'],
            'erstellt_am' => $bestellung['erstellt_am'],
            'bearbeitet_am' => $bestellung['bearbeitet_am'],
            'bearbeitet_von' => $bestellung['bearbeitet_von'],
            'notizen' => $bestellung['notizen'],
            'archiviert_von' => $archiviertVon
        ]);

        // Aus Haupttabelle löschen
        $stmt = $pdo->prepare("DELETE FROM bestellungen WHERE id = ?");
        $stmt->execute([$id]);

        $pdo->commit();
        return true;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Archivierung fehlgeschlagen: " . $e->getMessage());
        return false;
    }
}

/**
 * Holt archivierte Bestellungen
 */
function getArchivedBestellungen(int $page = 1, int $limit = ITEMS_PER_PAGE): array {
    $pdo = Database::getInstance();
    $offset = ($page - 1) * $limit;
    $sql = "SELECT * FROM bestellungen_archiv ORDER BY archiviert_am DESC LIMIT $limit OFFSET $offset";
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll();
}

/**
 * Zählt archivierte Bestellungen
 */
function countArchivedBestellungen(): int {
    $pdo = Database::getInstance();
    $stmt = $pdo->query("SELECT COUNT(*) FROM bestellungen_archiv");
    return (int) $stmt->fetchColumn();
}

/**
 * Erstellt eine neue Bestellung mit Status 'warte_auf_genehmigung'
 */
function createBestellungWithStatus(array $data): int {
    $pdo = Database::getInstance();

    $sql = "INSERT INTO bestellungen (
        schul_name, software_name, version, hersteller, hersteller_website, isbn,
        software_quelle, download_pfad, lizenz_typ, lizenzschluessel, lizenzserver_anmerkungen, status
    ) VALUES (
        :schul_name, :software_name, :version, :hersteller, :hersteller_website, :isbn,
        :software_quelle, :download_pfad, :lizenz_typ, :lizenzschluessel, :lizenzserver_anmerkungen, 'warte_auf_genehmigung'
    )";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'schul_name' => $data['schul_name'],
        'software_name' => $data['software_name'],
        'version' => $data['version'],
        'hersteller' => $data['hersteller'],
        'hersteller_website' => $data['hersteller_website'],
        'isbn' => $data['isbn'] ?? null,
        'software_quelle' => $data['software_quelle'],
        'download_pfad' => $data['download_pfad'] ?? null,
        'lizenz_typ' => $data['lizenz_typ'],
        'lizenzschluessel' => $data['lizenzschluessel'] ?? null,
        'lizenzserver_anmerkungen' => $data['lizenzserver_anmerkungen'] ?? null
    ]);

    return (int) $pdo->lastInsertId(DB_TYPE === 'pgsql' ? 'bestellungen_id_seq' : null);
}
