<?php
/**
 * E-Mail-Funktionen (SMTP)
 */

require_once __DIR__ . '/config.php';

/**
 * Sendet eine E-Mail via SMTP
 */
function send_email(string $to, string $subject, string $body, ?string $attachmentPath = null, ?string $cc = null): bool {
    // Für einfache Implementierung ohne PHPMailer verwenden wir socket-basiertes SMTP
    // In Produktion empfehle ich PHPMailer zu installieren

    try {
        $headers = [];
        $headers[] = "From: " . SMTP_FROM_NAME . " <" . SMTP_FROM_EMAIL . ">";
        $headers[] = "Reply-To: " . SMTP_FROM_EMAIL;
        $headers[] = "MIME-Version: 1.0";

        if ($cc) {
            $headers[] = "Cc: $cc";
        }

        if ($attachmentPath && file_exists($attachmentPath)) {
            // E-Mail mit Anhang
            $boundary = md5(time());
            $headers[] = "Content-Type: multipart/mixed; boundary=\"$boundary\"";

            $message = "--$boundary\r\n";
            $message .= "Content-Type: text/plain; charset=UTF-8\r\n";
            $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
            $message .= $body . "\r\n\r\n";

            // Anhang
            $fileContent = file_get_contents($attachmentPath);
            $fileContent = chunk_split(base64_encode($fileContent));
            $fileName = basename($attachmentPath);

            $message .= "--$boundary\r\n";
            $message .= "Content-Type: application/pdf; name=\"$fileName\"\r\n";
            $message .= "Content-Disposition: attachment; filename=\"$fileName\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n\r\n";
            $message .= $fileContent . "\r\n";
            $message .= "--$boundary--";

            $body = $message;
        } else {
            $headers[] = "Content-Type: text/plain; charset=UTF-8";
        }

        // SMTP-Versand (CC-Empfänger als zusätzlichen Envelope-Empfänger übergeben)
        $result = smtp_send($to, $subject, $body, implode("\r\n", $headers), $cc);

        if ($result) {
            log_email($to, $subject, 'SUCCESS');
            return true;
        } else {
            log_email($to, $subject, 'FAILED');
            return false;
        }
    } catch (Exception $e) {
        error_log("E-Mail-Fehler: " . $e->getMessage());
        log_email($to, $subject, 'ERROR: ' . $e->getMessage());
        return false;
    }
}

/**
 * SMTP-Versand über Socket
 */
function smtp_send(string $to, string $subject, string $body, string $headers, ?string $cc = null): bool {
    $smtp = @fsockopen(
        (SMTP_SECURE === 'ssl' ? 'ssl://' : '') . SMTP_HOST,
        SMTP_PORT,
        $errno,
        $errstr,
        30
    );

    if (!$smtp) {
        error_log("SMTP-Verbindung fehlgeschlagen: $errstr ($errno)");
        return false;
    }

    // Hilfsfunktion für SMTP-Kommunikation
    $get_response = function() use ($smtp) {
        $response = '';
        while ($line = fgets($smtp, 515)) {
            $response .= $line;
            if (substr($line, 3, 1) == ' ') break;
        }
        return $response;
    };

    $send_command = function($command) use ($smtp, $get_response) {
        fputs($smtp, $command . "\r\n");
        return $get_response();
    };

    try {
        $get_response(); // Willkommensnachricht

        // EHLO
        $response = $send_command("EHLO " . gethostname());
        if (substr($response, 0, 3) != '250') {
            throw new Exception("EHLO fehlgeschlagen: $response");
        }

        // STARTTLS wenn TLS
        if (SMTP_SECURE === 'tls') {
            $response = $send_command("STARTTLS");
            if (substr($response, 0, 3) != '220') {
                throw new Exception("STARTTLS fehlgeschlagen: $response");
            }
            stream_socket_enable_crypto($smtp, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
            $send_command("EHLO " . gethostname());
        }

        // AUTH LOGIN
        $response = $send_command("AUTH LOGIN");
        if (substr($response, 0, 3) != '334') {
            throw new Exception("AUTH LOGIN fehlgeschlagen: $response");
        }

        $response = $send_command(base64_encode(SMTP_USER));
        if (substr($response, 0, 3) != '334') {
            throw new Exception("Username fehlgeschlagen: $response");
        }

        $response = $send_command(base64_encode(SMTP_PASS));
        if (substr($response, 0, 3) != '235') {
            throw new Exception("Passwort fehlgeschlagen: $response");
        }

        // MAIL FROM
        $response = $send_command("MAIL FROM:<" . SMTP_FROM_EMAIL . ">");
        if (substr($response, 0, 3) != '250') {
            throw new Exception("MAIL FROM fehlgeschlagen: $response");
        }

        // RCPT TO (Hauptempfänger)
        $response = $send_command("RCPT TO:<$to>");
        if (substr($response, 0, 3) != '250') {
            throw new Exception("RCPT TO fehlgeschlagen: $response");
        }

        // RCPT TO (CC-Empfänger, falls vorhanden)
        if ($cc) {
            $ccAddresses = array_map('trim', explode(',', $cc));
            foreach ($ccAddresses as $ccAddr) {
                if (!empty($ccAddr)) {
                    $response = $send_command("RCPT TO:<$ccAddr>");
                    if (substr($response, 0, 3) != '250') {
                        error_log("RCPT TO CC fehlgeschlagen für $ccAddr: $response");
                    }
                }
            }
        }

        // DATA
        $response = $send_command("DATA");
        if (substr($response, 0, 3) != '354') {
            throw new Exception("DATA fehlgeschlagen: $response");
        }

        // E-Mail-Inhalt
        $email = "Subject: $subject\r\n";
        $email .= "To: $to\r\n";
        $email .= $headers . "\r\n\r\n";
        $email .= $body;
        $email .= "\r\n.";

        $response = $send_command($email);
        if (substr($response, 0, 3) != '250') {
            throw new Exception("Nachricht fehlgeschlagen: $response");
        }

        $send_command("QUIT");
        fclose($smtp);
        return true;

    } catch (Exception $e) {
        error_log("SMTP-Fehler: " . $e->getMessage());
        @fclose($smtp);
        return false;
    }
}

/**
 * Sendet Ticket-E-Mail bei neuer Bestellung (mit konfigurierbarem Template)
 */
function send_ticket_email(int $bestellungId, array $data): bool {
    require_once __DIR__ . '/functions.php';

    $templateData = array_merge($data, ['id' => $bestellungId]);

    $subject = replace_mail_placeholders(MAIL_TEMPLATE_TICKET_SUBJECT, $templateData);
    $body = replace_mail_placeholders(MAIL_TEMPLATE_TICKET_BODY, $templateData);

    return send_email(MAIL_TICKET_TO, $subject, $body);
}

/**
 * Sendet E-Mail an iServ mit PDF-Anhang (mit konfigurierbarem Template)
 */
function send_iserv_email(array $bestellung, string $pdfPath): bool {
    require_once __DIR__ . '/functions.php';

    $subject = replace_mail_placeholders(MAIL_TEMPLATE_ISERV_SUBJECT, $bestellung);
    $body = replace_mail_placeholders(MAIL_TEMPLATE_ISERV_BODY, $bestellung);

    // CC-Empfänger aus Konfiguration
    $cc = defined('MAIL_COPY_TO') && !empty(MAIL_COPY_TO) ? MAIL_COPY_TO : null;

    return send_email(MAIL_ISERV_TO, $subject, $body, $pdfPath, $cc);
}

/**
 * Loggt E-Mail-Versand
 */
function log_email(string $to, string $subject, string $status): void {
    $message = date('Y-m-d H:i:s') . " [$status] To: $to, Subject: $subject\n";
    @file_put_contents(LOG_PATH . 'email.log', $message, FILE_APPEND | LOCK_EX);
}
