<?php
/**
 * Hilfsfunktionen für das Softwarebestell-System
 */

/**
 * Escapet HTML-Ausgaben
 */
function h(string|null $string): string {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Validiert eine URL
 */
function validate_url(string $url): bool {
    return filter_var($url, FILTER_VALIDATE_URL) !== false;
}

/**
 * Validiert eine E-Mail-Adresse
 */
function validate_email(string $email): bool {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Prüft CSRF-Token
 */
function verify_csrf_token(string $token): bool {
    if (empty($_SESSION['csrf_token'])) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Generiert ein neues CSRF-Token-Feld für Formulare
 */
function csrf_field(): string {
    return '<input type="hidden" name="csrf_token" value="' . h($_SESSION['csrf_token']) . '">';
}

/**
 * Übersetzt Status-Codes in lesbare Labels
 */
function status_label(string $status): string {
    return match($status) {
        'warte_auf_genehmigung' => 'Warte auf Genehmigung',
        'genehmigt' => 'Genehmigt',
        'gesendet' => 'An iServ gesendet',
        'erledigt' => 'Erledigt',
        'abgelehnt' => 'Abgelehnt',
        // Legacy-Status für bestehende Daten
        'neu' => 'Neu',
        'in_pruefung' => 'In Prüfung',
        'korrektur_notwendig' => 'Korrektur notwendig',
        'versendet' => 'An iServ gesendet',
        default => $status
    };
}

/**
 * Gibt CSS-Klasse für Status zurück
 */
function status_class(string $status): string {
    return match($status) {
        'warte_auf_genehmigung' => 'status-pending',
        'genehmigt' => 'status-approved',
        'gesendet' => 'status-sent',
        'erledigt' => 'status-completed',
        'abgelehnt' => 'status-rejected',
        // Legacy-Status
        'neu' => 'status-new',
        'in_pruefung' => 'status-review',
        'korrektur_notwendig' => 'status-correction',
        'versendet' => 'status-sent',
        default => ''
    };
}

/**
 * Übersetzt Software-Quelle in lesbaren Text
 */
function source_label(string $source): string {
    return match($source) {
        'downloadFree' => 'Kostenfreier Download',
        'downloadCustomer' => 'Download vom Kundenserver',
        'physicalMedia' => 'Datenträger',
        default => $source
    };
}

/**
 * Übersetzt Lizenz-Typ in lesbaren Text
 */
function license_label(string $type): string {
    return match($type) {
        'noLicense' => 'Keine Lizenz notwendig',
        'licenseKey' => 'Lizenzschlüssel',
        'licenseHardware' => 'Hardware-Dongle',
        'licenseServer' => 'Lizenzserver',
        default => $type
    };
}

/**
 * Formatiert Datum für Anzeige
 */
function format_date(?string $date, string $format = 'd.m.Y H:i'): string {
    if (empty($date)) {
        return '-';
    }
    $dt = new DateTime($date);
    return $dt->format($format);
}

/**
 * Validiert die Formular-Eingaben für eine Bestellung
 */
function validate_bestellung(array $data): array {
    $errors = [];

    // Pflichtfelder
    $required = [
        'schulName' => 'Name der Schule',
        'softwareName' => 'Name der Software',
        'version' => 'Versionsnummer',
        'manufacturer' => 'Hersteller',
        'website' => 'Hersteller-Website',
        'softwareSource' => 'Bezugsquelle',
        'licenseType' => 'Lizenztyp'
    ];

    foreach ($required as $field => $label) {
        if (empty($data[$field])) {
            $errors[] = "$label ist ein Pflichtfeld.";
        }
    }

    // Checkbox "verified" prüfen
    if (empty($data['verified'])) {
        $errors[] = "Bitte bestätigen Sie, dass die Software noch nicht paketiert wurde.";
    }

    // URL validieren
    if (!empty($data['website']) && !validate_url($data['website'])) {
        $errors[] = "Bitte geben Sie eine gültige URL für die Hersteller-Website ein.";
    }

    // Software-Quelle: Download-Pfad erforderlich wenn downloadFree/downloadCustomer
    if (in_array($data['softwareSource'] ?? '', ['downloadFree', 'downloadCustomer'])) {
        $pathField = $data['softwareSource'] === 'downloadFree' ? 'downloadFreePath' : 'downloadCustomerPath';
        if (empty($data[$pathField])) {
            $errors[] = "Bitte geben Sie den Download-Pfad an.";
        }
    }

    // Lizenztyp: Schlüssel/Anmerkungen erforderlich
    if (($data['licenseType'] ?? '') === 'licenseKey' && empty($data['licenseKeyValue'])) {
        $errors[] = "Bitte geben Sie den Lizenzschlüssel ein.";
    }

    return $errors;
}

/**
 * Bereitet Formulardaten für die Datenbank vor
 */
function prepare_bestellung_data(array $post): array {
    $downloadPfad = null;
    if ($post['softwareSource'] === 'downloadFree') {
        $downloadPfad = $post['downloadFreePath'] ?? null;
    } elseif ($post['softwareSource'] === 'downloadCustomer') {
        $downloadPfad = $post['downloadCustomerPath'] ?? null;
    }

    return [
        'schul_name' => trim($post['schulName']),
        'software_name' => trim($post['softwareName']),
        'version' => trim($post['version']),
        'hersteller' => trim($post['manufacturer']),
        'hersteller_website' => trim($post['website']),
        'isbn' => !empty($post['isbn']) ? trim($post['isbn']) : null,
        'software_quelle' => $post['softwareSource'],
        'download_pfad' => $downloadPfad ? trim($downloadPfad) : null,
        'lizenz_typ' => $post['licenseType'],
        'lizenzschluessel' => !empty($post['licenseKeyValue']) ? trim($post['licenseKeyValue']) : null,
        'lizenzserver_anmerkungen' => !empty($post['licenseServerNotes']) ? trim($post['licenseServerNotes']) : null
    ];
}

/**
 * Loggt allgemeine Anwendungsereignisse
 */
function app_log(string $message, string $level = 'INFO'): void {
    $logMessage = date('Y-m-d H:i:s') . " [$level] $message\n";
    @file_put_contents(LOG_PATH . 'app.log', $logMessage, FILE_APPEND | LOCK_EX);
}

/**
 * Setzt Flash-Nachricht für die nächste Seite
 */
function set_flash(string $message, string $type = 'success'): void {
    $_SESSION['flash'] = ['message' => $message, 'type' => $type];
}

/**
 * Holt und löscht Flash-Nachricht
 */
function get_flash(): ?array {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

/**
 * Gibt Alle verfügbaren Status zurück
 */
function get_all_statuses(): array {
    return [
        'warte_auf_genehmigung' => 'Warte auf Genehmigung',
        'genehmigt' => 'Genehmigt',
        'gesendet' => 'An iServ gesendet',
        'erledigt' => 'Erledigt',
        'abgelehnt' => 'Abgelehnt'
    ];
}

/**
 * Ersetzt Platzhalter in E-Mail-Templates
 */
function replace_mail_placeholders(string $template, array $data): string {
    $replacements = [
        '{id}' => $data['id'] ?? '',
        '{software_name}' => $data['software_name'] ?? '',
        '{version}' => $data['version'] ?? '',
        '{schul_name}' => $data['schul_name'] ?? '',
        '{hersteller}' => $data['hersteller'] ?? '',
        '{datum}' => date('d.m.Y H:i:s'),
        '{admin_link}' => (isset($_SERVER['HTTP_HOST']) ?
            ((isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST']) : '')
            . ADMIN_URL . '/detail.php?id=' . ($data['id'] ?? ''),
        '{company_name}' => defined('COMPANY_NAME') ? COMPANY_NAME : 'Softwarebestell-System'
    ];

    return str_replace(array_keys($replacements), array_values($replacements), $template);
}
