<?php
/**
 * PDF-Generator für iServ-Softwarebestellungen
 * Layout basiert auf Info-Software.pdf (Original iServ-Vorlage)
 */

require_once FPDF_PATH . 'fpdf.php';

class IServPDF extends FPDF {
    private array $data;
    private string $logoPath;

    // Seitenränder
    private float $marginLeft = 25;
    private float $marginRight = 25;
    private float $contentWidth;

    public function __construct(array $data) {
        parent::__construct('P', 'mm', 'A4');
        $this->data = $data;
        $this->logoPath = $_SERVER['DOCUMENT_ROOT'] . '/logo.png';
        $this->contentWidth = 210 - $this->marginLeft - $this->marginRight;
        $this->SetMargins($this->marginLeft, 15, $this->marginRight);
        $this->SetAutoPageBreak(true, 20);
    }

    /**
     * Konvertiert UTF-8 zu ISO-8859-1 für FPDF
     */
    private function utf8(string $text): string {
        return iconv('UTF-8', 'ISO-8859-1//TRANSLIT', $text);
    }

    /**
     * PDF-Header: Logo oben rechts + horizontale Linie
     */
    function Header(): void {
        // Logo oben rechts
        if (file_exists($this->logoPath)) {
            $this->Image($this->logoPath, 150, 10, 40);
        }

        // Horizontale Linie unter Logo-Bereich
        $this->SetY(35);
        $this->SetDrawColor(0, 102, 204);
        $this->SetLineWidth(0.8);
        $this->Line($this->marginLeft, 35, 210 - $this->marginRight, 35);

        $this->SetY(42);
    }

    /**
     * PDF-Footer
     */
    function Footer(): void {
        $this->SetY(-15);
        $this->SetFont('Arial', 'I', 8);
        $this->SetTextColor(150, 150, 150);
        $this->Cell(0, 10, $this->utf8('Generiert am ' . date('d.m.Y H:i') . ' | Seite ' . $this->PageNo()), 0, 0, 'C');
    }

    /**
     * Zeichnet eine Checkbox (Quadrat, bei checked mit Kreuz)
     */
    private function drawCheckbox(float $x, float $y, bool $checked = false): void {
        $size = 3.5;
        $this->SetDrawColor(0, 0, 0);
        $this->SetLineWidth(0.25);
        $this->Rect($x, $y, $size, $size);

        if ($checked) {
            // Kreuz zeichnen
            $this->SetLineWidth(0.4);
            $this->Line($x + 0.5, $y + 0.5, $x + $size - 0.5, $y + $size - 0.5);
            $this->Line($x + $size - 0.5, $y + 0.5, $x + 0.5, $y + $size - 0.5);
            $this->SetLineWidth(0.25);
        }
    }

    /**
     * Zeichnet ein Textfeld mit Label und Unterstrich-Linie für den Wert
     */
    private function labelField(string $label, ?string $value): void {
        $this->SetFont('Arial', '', 10);
        $this->Cell(0, 6, $this->utf8($label . ' ' . ($value ?? '')), 0, 1, 'L');
        // Unterstrich-Linie für das Eingabefeld
        $y = $this->GetY();
        $this->SetDrawColor(180, 180, 180);
        $this->SetLineWidth(0.2);
        $this->Line($this->marginLeft, $y, 210 - $this->marginRight, $y);
        $this->Ln(3);
    }

    /**
     * Zeichnet einen Abschnittstitel (fett, wie im Original)
     */
    private function sectionTitle(string $title, ?string $subtitle = null): void {
        $this->Ln(4);
        $this->SetFont('Arial', 'B', 10);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 6, $this->utf8($title), 0, 1, 'L');

        if ($subtitle) {
            $this->SetFont('Arial', '', 9);
            $this->Cell(0, 5, $this->utf8($subtitle), 0, 1, 'L');
        }
        $this->Ln(2);
    }

    /**
     * Zeichnet eine Checkbox-Option mit Text (eingerückt)
     */
    private function checkboxOption(string $text, bool $checked, ?string $inputValue = null, bool $hasInputLine = false): void {
        $indent = 8;
        $x = $this->marginLeft + $indent;
        $y = $this->GetY();

        $this->drawCheckbox($x, $y + 0.5, $checked);

        $this->SetX($x + 6);
        $this->SetFont('Arial', '', 10);
        $this->MultiCell($this->contentWidth - $indent - 6, 5, $this->utf8($text), 0, 'L');

        // Wenn es ein Eingabefeld gibt (z.B. Download-Pfad, Lizenzschlüssel)
        if ($hasInputLine || ($inputValue !== null && $inputValue !== '')) {
            $this->SetX($x + 6);
            if ($checked && !empty($inputValue)) {
                $this->SetFont('Arial', '', 10);
                $this->Cell($this->contentWidth - $indent - 6, 6, $this->utf8($inputValue), 'B', 1, 'L');
            } else {
                // Leere Eingabezeile
                $lineY = $this->GetY() + 5;
                $this->SetDrawColor(180, 180, 180);
                $this->SetLineWidth(0.2);
                $this->Line($x + 6, $lineY, 210 - $this->marginRight, $lineY);
                $this->Ln(7);
            }
        }

        $this->Ln(2);
    }

    /**
     * Generiert das komplette PDF im Original-Layout
     */
    function Generate(): void {
        $this->AddPage();

        // Titel
        $this->SetTextColor(0, 0, 0);
        $this->SetFont('Arial', 'B', 13);
        $this->Cell(0, 7, $this->utf8('NEUE SOFTWAREPAKETE'), 0, 1, 'L');
        $this->Ln(2);

        // Untertitel
        $this->SetFont('Arial', 'B', 10);
        $this->Cell(0, 5, $this->utf8('Angaben zur Erstellung eines neuen Softwarepaketes'), 0, 1, 'L');
        $this->SetFont('Arial', '', 9);
        $this->Cell(0, 5, $this->utf8('(Eine Liste aller aktuellen Softwarepakete der IServ GmbH finden Sie hier: iserv.eu/deploy/software)'), 0, 1, 'L');
        $this->Ln(4);

        // Prüfhinweis-Checkbox
        $checkX = $this->marginLeft + 8;
        $checkY = $this->GetY() + 0.5;
        $this->drawCheckbox($checkX, $checkY, true);
        $this->SetX($checkX + 6);
        $this->SetFont('Arial', '', 10);
        $this->Cell(0, 5, $this->utf8('Ich habe geprüft, dass die Software noch nicht von IServ paketiert wurde.'), 0, 1, 'L');
        $this->Ln(5);

        // Schulname (Zusatzfeld unseres Systems)
        $this->SetFont('Arial', 'B', 10);
        $this->Cell(45, 6, $this->utf8('Schule:'), 0, 0, 'L');
        $this->SetFont('Arial', '', 10);
        $this->Cell(0, 6, $this->utf8($this->data['schul_name'] ?? ''), 'B', 1, 'L');
        $this->Ln(3);

        // Grundinformationen - Felder wie im Original
        $this->labelField('Name der Software:', $this->data['software_name'] ?? '');
        $this->labelField('Versionsnummer:', $this->data['version'] ?? '');
        $this->labelField('Hersteller:', $this->data['hersteller'] ?? '');
        $this->labelField('Hersteller-Website:', $this->data['hersteller_website'] ?? '');
        $this->labelField('ISBN (sinnvoll für Lernsoftware):', $this->data['isbn'] ?? '');

        // Bezug der Software
        $source = $this->data['software_quelle'] ?? '';
        $downloadPath = $this->data['download_pfad'] ?? '';

        $this->sectionTitle('Bezug der Software');

        $this->checkboxOption(
            'Kostenfreier Download über Herstellerseite (Pfad zur Datei angeben, ohne Registrierung)',
            $source === 'downloadFree',
            $source === 'downloadFree' ? $downloadPath : null,
            true
        );

        $this->checkboxOption(
            'Bereitstellung als Download vom Kundenserver (Pfad zur Datei angeben)',
            $source === 'downloadCustomer',
            $source === 'downloadCustomer' ? $downloadPath : null,
            true
        );

        $this->checkboxOption(
            'Software wird auf Datenträger bereitgestellt',
            $source === 'physicalMedia'
        );

        // Lizenzierung der Software
        $licenseType = $this->data['lizenz_typ'] ?? '';

        $this->sectionTitle(
            'Lizenzierung der Software',
            '(Lizenzinformationen sind für IServ zum Testen der Paketierung zwingend erforderlich)'
        );

        $this->checkboxOption(
            'keine Lizenz notwendig',
            $licenseType === 'noLicense'
        );

        $this->checkboxOption(
            'Lizenzschlüssel:',
            $licenseType === 'licenseKey',
            $licenseType === 'licenseKey' ? ($this->data['lizenzschluessel'] ?? '') : null,
            true
        );

        $this->checkboxOption(
            'Lizenz wird als Hardware bereitgestellt (z. B. Dongle)',
            $licenseType === 'licenseHardware'
        );

        $this->checkboxOption(
            'Lizenzserver wird benötigt (voraussichtlich höherer Entwicklungsaufwand)',
            $licenseType === 'licenseServer'
        );

        // Anmerkungen zum Lizenzserver (immer anzeigen, wie im Original)
        $indent = 8;
        $this->SetX($this->marginLeft + $indent + 6);
        $this->SetFont('Arial', '', 10);
        $serverNotes = ($licenseType === 'licenseServer') ? ($this->data['lizenzserver_anmerkungen'] ?? '') : '';
        $this->Cell(0, 5, $this->utf8('Anmerkungen zum Lizenzserver: ' . $serverNotes), 0, 1, 'L');
        // Eingabelinie
        $lineY = $this->GetY() + 2;
        $this->SetDrawColor(180, 180, 180);
        $this->SetLineWidth(0.2);
        $this->Line($this->marginLeft + $indent + 6, $lineY, 210 - $this->marginRight, $lineY);
        $this->Ln(8);

        // Interne Notizen (nur wenn vorhanden - Zusatz unseres Systems)
        if (!empty($this->data['notizen'])) {
            $this->sectionTitle('Interne Notizen');
            $this->SetFont('Arial', '', 10);
            $this->MultiCell(0, 5, $this->utf8($this->data['notizen']), 0, 'L');
        }
    }

    /**
     * Speichert das PDF in eine Datei
     */
    function SaveToFile(string $filename): string {
        $this->Generate();
        $filepath = PDF_PATH . $filename;
        $this->Output('F', $filepath);
        return $filepath;
    }

    /**
     * Gibt das PDF direkt im Browser aus
     */
    function OutputToBrowser(string $filename): void {
        $this->Generate();
        $this->Output('D', $filename);
    }
}

/**
 * Generiert ein PDF für eine Bestellung
 */
function generate_pdf(array $bestellung): string {
    $pdf = new IServPDF($bestellung);
    $filename = 'Softwarepaket_' . $bestellung['id'] . '_' . preg_replace('/[^a-zA-Z0-9]/', '_', $bestellung['software_name']) . '.pdf';
    return $pdf->SaveToFile($filename);
}

/**
 * Gibt ein PDF direkt zum Download aus
 */
function output_pdf(array $bestellung): void {
    $pdf = new IServPDF($bestellung);
    $filename = 'Softwarepaket_' . $bestellung['id'] . '_' . preg_replace('/[^a-zA-Z0-9]/', '_', $bestellung['software_name']) . '.pdf';
    $pdf->OutputToBrowser($filename);
}
