/**
 * SecNote App Module
 * Handles Create-Flow, View-Flow, and UI interactions
 */
(function() {
    'use strict';

    var C = SecNoteCrypto;
    var CFG = window.SECNOTE_CONFIG;
    var MSG = CFG.messages;

    // --- Browser Check ---
    if (typeof forge === 'undefined') {
        document.getElementById('haupt').innerHTML =
            '<h3>secnote.eblok.de</h3><p>' + MSG.browser_nicht_unterstuetzt + '</p>';
        return;
    }

    // --- Utility ---

    function escapeHtml(str) {
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }

    function readFileAsArrayBuffer(file) {
        return new Promise(function(resolve, reject) {
            var reader = new FileReader();
            reader.onload = function() { resolve(new Uint8Array(reader.result)); };
            reader.onerror = function() { reject(reader.error); };
            reader.readAsArrayBuffer(file);
        });
    }

    // --- Create-Flow ---

    var form = document.getElementById('secnote-form');
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            handleCreate();
        });
    }

    async function handleCreate() {
        var textEl = document.getElementById('secnote-text');
        var bildEl = document.getElementById('secnote-bild');
        var pwEl = document.getElementById('pw');
        var fehlerEl = document.getElementById('secnote-fehler');
        var submitBtn = document.getElementById('secnote-submit');
        var resultDiv = document.getElementById('secnote-result');
        var formDiv = document.getElementById('schreiben');

        var text = textEl ? textEl.value : '';
        var password = pwEl ? pwEl.value : '';
        var fehler = fehlerEl ? fehlerEl.value : '0';
        var bildFile = (bildEl && bildEl.files.length > 0) ? bildEl.files[0] : null;

        // Validate: at least text or image
        if (text.trim() === '' && !bildFile) {
            alert(MSG.eingabe_leer);
            return;
        }

        // Validate image size
        if (bildFile && bildFile.size > CFG.maxImageSize) {
            alert(MSG.bild_zu_gross);
            return;
        }

        // Disable form
        submitBtn.disabled = true;
        submitBtn.value = MSG.wird_verschluesselt;
        form.classList.add('secnote-processing');

        try {
            var key = await C.generateKey();
            var keyStr = await C.exportKeyToBase64url(key);

            // Encrypt text
            var encryptedMsg = '';
            if (text.trim() !== '') {
                var textBytes = C.stringToBytes(text);
                var encBytes;
                if (password) {
                    encBytes = await C.encryptWithPassword(textBytes, key, password);
                } else {
                    encBytes = await C.encrypt(textBytes, key);
                }
                encryptedMsg = C.base64Encode(encBytes);
            }

            // Encrypt image
            var encryptedImg = '';
            if (bildFile) {
                var imgRaw = await readFileAsArrayBuffer(bildFile);
                var imgPayload = JSON.stringify({
                    mime: bildFile.type,
                    data: C.base64Encode(imgRaw)
                });
                var imgBytes = C.stringToBytes(imgPayload);
                var encImgBytes;
                if (password) {
                    encImgBytes = await C.encryptWithPassword(imgBytes, key, password);
                } else {
                    encImgBytes = await C.encrypt(imgBytes, key);
                }
                encryptedImg = C.base64Encode(encImgBytes);
            }

            // Send to server
            var fd = new FormData();
            fd.append('msg', encryptedMsg);
            fd.append('img', encryptedImg);
            fd.append('passwort', password ? '1' : '0');
            fd.append('fehler', fehler);
            fd.append('csrf_token', CFG.csrfToken);

            var resp = await fetch('index.php?cont=api_create', {
                method: 'POST',
                body: fd
            });
            var data = await resp.json();

            if (data.success) {
                var fullUrl = data.url + '#' + keyStr;
                // Show result
                formDiv.style.display = 'none';
                resultDiv.style.display = '';
                document.getElementById('url').textContent = fullUrl;
                document.getElementById('hinweis').innerHTML = MSG.hinweis;
                // Init clipboard
                if (typeof ClipboardJS !== 'undefined') {
                    new ClipboardJS('.btn');
                } else if (typeof Clipboard !== 'undefined') {
                    new Clipboard('.btn');
                }
            } else {
                alert(data.error || 'Fehler beim Speichern.');
                submitBtn.disabled = false;
                submitBtn.value = MSG.link_erstellen;
                form.classList.remove('secnote-processing');
            }
        } catch (err) {
            console.error('Encryption error:', err);
            alert(MSG.entschluesselung_fehler);
            submitBtn.disabled = false;
            submitBtn.value = MSG.link_erstellen;
            form.classList.remove('secnote-processing');
        }
    }

    // --- View-Flow ---

    var viewerContainer = document.getElementById('viewer-container');
    if (viewerContainer) {
        handleView();
    }

    async function handleView() {
        var loadingDiv = document.getElementById('viewer-loading');
        var contentDiv = document.getElementById('viewer-content');
        var errorDiv = document.getElementById('viewer-error');
        var passwordDiv = document.getElementById('viewer-password');

        // Get ID and key from URL
        var params = new URLSearchParams(window.location.search);
        var id = params.get('id');
        var hash = window.location.hash.substring(1); // remove #

        if (!hash) {
            showError(MSG.schluessel_fehlt);
            return;
        }

        try {
            // Fetch encrypted data
            var resp = await fetch('index.php?cont=api_fetch&id=' + encodeURIComponent(id));
            var data = await resp.json();

            if (!data.found) {
                showError(MSG.keine_Nachricht);
                return;
            }

            var key = await C.importKeyFromBase64url(hash);

            if (data.passwort === 1) {
                // Show password form
                showPasswordForm(data, key, id);
            } else {
                // Decrypt directly
                await decryptAndShow(data, key, id, null);
            }
        } catch (err) {
            console.error('View error:', err);
            showError(MSG.entschluesselung_fehler);
        }

        function showError(msg) {
            loadingDiv.style.display = 'none';
            errorDiv.style.display = '';
            errorDiv.innerHTML = msg;
        }

        function showPasswordForm(data, key, id) {
            loadingDiv.style.display = 'none';
            passwordDiv.style.display = '';
            passwordDiv.innerHTML =
                '<span id="empfaengerhinweis">' + MSG.empfaengerhinweis + '</span>' +
                '<label for="pw">Passwort</label> ' +
                '<input id="pw" type="password" size="8" />' +
                ' <input id="check" type="checkbox" />' +
                '<label for="check"> ' + MSG.zeigen + '</label><br />' +
                '<span id="passworthinweis">' + MSG.passworthinweisEmpf + '</span><br />' +
                (data.fehler > 0 ? '<u>' + MSG.versuche_uebrig.replace('{n}', data.fehler) + '</u><br /><br />' : '') +
                '<input type="submit" id="pw-submit" value="' + MSG.zeigen + '" />';

            var checkBox = document.getElementById('check');
            var pwInput = document.getElementById('pw');
            checkBox.onclick = function() {
                pwInput.type = this.checked ? 'text' : 'password';
            };

            document.getElementById('pw-submit').addEventListener('click', async function() {
                var pw = document.getElementById('pw').value;
                if (!pw) return;
                this.disabled = true;
                this.value = MSG.wird_entschluesselt;
                try {
                    await decryptAndShow(data, key, id, pw);
                } catch (err) {
                    console.error('Decrypt error:', err);
                    // Wrong password - report attempt
                    try {
                        var aResp = await fetch('index.php?cont=api_attempt', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: 'id=' + encodeURIComponent(id) + '&csrf_token=' + encodeURIComponent(CFG.csrfToken)
                        });
                        var aData = await aResp.json();
                        if (aData.deleted) {
                            passwordDiv.style.display = 'none';
                            showError(MSG.passwort_falsch + '<br />' + MSG.nachricht_geloescht);
                        } else {
                            document.getElementById('pw-submit').disabled = false;
                            document.getElementById('pw-submit').value = MSG.zeigen;
                            data.fehler = aData.remaining;
                            // Update attempts display
                            showPasswordForm(data, key, id);
                        }
                    } catch (attemptErr) {
                        showError(MSG.entschluesselung_fehler);
                    }
                }
            });
        }

        async function decryptAndShow(data, key, id, password) {
            loadingDiv.style.display = 'none';
            passwordDiv.style.display = 'none';

            var html = '<h2>Deine Nachricht... </h2>';

            // Decrypt text
            if (data.msg && data.msg !== '') {
                var msgBytes = C.base64Decode(data.msg);
                var plainBytes;
                if (password) {
                    plainBytes = await C.decryptWithPassword(msgBytes, key, password);
                } else {
                    plainBytes = await C.decrypt(msgBytes, key);
                }
                var plaintext = C.bytesToString(plainBytes);
                html += escapeHtml(plaintext).replace(/\n/g, '<br />');
            }

            // Decrypt image
            if (data.img && data.img !== '') {
                var imgBytes = C.base64Decode(data.img);
                var imgPlainBytes;
                if (password) {
                    imgPlainBytes = await C.decryptWithPassword(imgBytes, key, password);
                } else {
                    imgPlainBytes = await C.decrypt(imgBytes, key);
                }
                var imgPayload = JSON.parse(C.bytesToString(imgPlainBytes));
                html += '<br /><img src="data:' + escapeHtml(imgPayload.mime) +
                    ';base64,' + imgPayload.data + '" style="max-width:100%;" />';
            }

            html += '<br /><br />' + MSG.geloescht;
            html += '<br /><br /><a href="index.php">' + MSG.selbstNachricht + '</a>';

            contentDiv.innerHTML = html;
            contentDiv.style.display = '';

            // Delete from server
            try {
                await fetch('index.php?cont=api_delete', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'id=' + encodeURIComponent(id) + '&csrf_token=' + encodeURIComponent(CFG.csrfToken)
                });
            } catch (delErr) {
                console.error('Delete error:', delErr);
            }
        }
    }
})();
