/**
 * SecNote Crypto Module
 * Client-side AES-256-GCM encryption via node-forge
 * Works on HTTP and HTTPS (no Secure Context required)
 */
const SecNoteCrypto = (function() {
    'use strict';

    var IV_LENGTH = 12;
    var SALT_LENGTH = 16;
    var KEY_LENGTH = 32;  // bytes (256 bits)
    var TAG_LENGTH = 16;  // bytes (128 bits)
    var PBKDF2_ITERATIONS = 600000;

    // --- forge binary string <-> Uint8Array ---

    function toStr(uint8arr) {
        var s = '';
        for (var i = 0; i < uint8arr.length; i++) {
            s += String.fromCharCode(uint8arr[i]);
        }
        return s;
    }

    function fromStr(str) {
        var arr = new Uint8Array(str.length);
        for (var i = 0; i < str.length; i++) {
            arr[i] = str.charCodeAt(i);
        }
        return arr;
    }

    function randomBytes(n) {
        return fromStr(forge.random.getBytesSync(n));
    }

    // --- Helpers ---

    function stringToBytes(str) {
        return new TextEncoder().encode(str);
    }

    function bytesToString(bytes) {
        return new TextDecoder().decode(bytes);
    }

    function base64urlEncode(bytes) {
        var arr = bytes instanceof Uint8Array ? bytes : new Uint8Array(bytes);
        var binary = '';
        for (var i = 0; i < arr.length; i++) {
            binary += String.fromCharCode(arr[i]);
        }
        return btoa(binary).replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
    }

    function base64urlDecode(str) {
        var s = str.replace(/-/g, '+').replace(/_/g, '/');
        while (s.length % 4 !== 0) { s += '='; }
        var binary = atob(s);
        var bytes = new Uint8Array(binary.length);
        for (var i = 0; i < binary.length; i++) {
            bytes[i] = binary.charCodeAt(i);
        }
        return bytes;
    }

    function base64Encode(bytes) {
        var arr = bytes instanceof Uint8Array ? bytes : new Uint8Array(bytes);
        var binary = '';
        for (var i = 0; i < arr.length; i++) {
            binary += String.fromCharCode(arr[i]);
        }
        return btoa(binary);
    }

    function base64Decode(str) {
        var binary = atob(str);
        var bytes = new Uint8Array(binary.length);
        for (var i = 0; i < binary.length; i++) {
            bytes[i] = binary.charCodeAt(i);
        }
        return bytes;
    }

    // --- Core Crypto (forge AES-256-GCM) ---
    // Key = Uint8Array(32), encrypt/decrypt return/accept Uint8Array

    async function generateKey() {
        return randomBytes(KEY_LENGTH);
    }

    async function exportKeyToBase64url(key) {
        return base64urlEncode(key);
    }

    async function importKeyFromBase64url(str) {
        return base64urlDecode(str);
    }

    async function encrypt(plaintextBytes, key) {
        var iv = randomBytes(IV_LENGTH);
        var cipher = forge.cipher.createCipher('AES-GCM', toStr(key));
        cipher.start({ iv: toStr(iv), tagLength: 128 });
        cipher.update(forge.util.createBuffer(toStr(plaintextBytes)));
        cipher.finish();

        var ciphertext = fromStr(cipher.output.getBytes());
        var tag = fromStr(cipher.mode.tag.getBytes());

        // Format: iv(12) + ciphertext + authTag(16)
        var result = new Uint8Array(IV_LENGTH + ciphertext.length + TAG_LENGTH);
        result.set(iv, 0);
        result.set(ciphertext, IV_LENGTH);
        result.set(tag, IV_LENGTH + ciphertext.length);
        return result;
    }

    async function decrypt(ciphertextBytes, key) {
        var iv = ciphertextBytes.slice(0, IV_LENGTH);
        var tag = ciphertextBytes.slice(ciphertextBytes.length - TAG_LENGTH);
        var data = ciphertextBytes.slice(IV_LENGTH, ciphertextBytes.length - TAG_LENGTH);

        var decipher = forge.cipher.createDecipher('AES-GCM', toStr(key));
        decipher.start({
            iv: toStr(iv),
            tag: forge.util.createBuffer(toStr(tag)),
            tagLength: 128
        });
        decipher.update(forge.util.createBuffer(toStr(data)));
        var pass = decipher.finish();

        if (!pass) {
            throw new Error('Decryption failed: authentication tag mismatch');
        }

        return fromStr(decipher.output.getBytes());
    }

    async function deriveKeyFromPassword(password, salt) {
        return new Promise(function(resolve, reject) {
            forge.pkcs5.pbkdf2(
                password, toStr(salt),
                PBKDF2_ITERATIONS, KEY_LENGTH, 'sha256',
                function(err, derivedKey) {
                    if (err) { reject(err); return; }
                    resolve(fromStr(derivedKey));
                }
            );
        });
    }

    async function encryptWithPassword(plaintextBytes, key, password) {
        // Inner encryption with random key
        var innerCiphertext = await encrypt(plaintextBytes, key);
        // Outer encryption with PBKDF2-derived key
        var salt = randomBytes(SALT_LENGTH);
        var pwKey = await deriveKeyFromPassword(password, salt);
        var iv = randomBytes(IV_LENGTH);

        var cipher = forge.cipher.createCipher('AES-GCM', toStr(pwKey));
        cipher.start({ iv: toStr(iv), tagLength: 128 });
        cipher.update(forge.util.createBuffer(toStr(innerCiphertext)));
        cipher.finish();

        var outerCiphertext = fromStr(cipher.output.getBytes());
        var tag = fromStr(cipher.mode.tag.getBytes());

        // Format: salt(16) + iv(12) + outerCiphertext + tag(16)
        var result = new Uint8Array(SALT_LENGTH + IV_LENGTH + outerCiphertext.length + TAG_LENGTH);
        result.set(salt, 0);
        result.set(iv, SALT_LENGTH);
        result.set(outerCiphertext, SALT_LENGTH + IV_LENGTH);
        result.set(tag, SALT_LENGTH + IV_LENGTH + outerCiphertext.length);
        return result;
    }

    async function decryptWithPassword(ciphertextBytes, key, password) {
        var salt = ciphertextBytes.slice(0, SALT_LENGTH);
        var iv = ciphertextBytes.slice(SALT_LENGTH, SALT_LENGTH + IV_LENGTH);
        var tag = ciphertextBytes.slice(ciphertextBytes.length - TAG_LENGTH);
        var data = ciphertextBytes.slice(SALT_LENGTH + IV_LENGTH, ciphertextBytes.length - TAG_LENGTH);

        // Outer decryption with PBKDF2-derived key
        var pwKey = await deriveKeyFromPassword(password, salt);

        var decipher = forge.cipher.createDecipher('AES-GCM', toStr(pwKey));
        decipher.start({
            iv: toStr(iv),
            tag: forge.util.createBuffer(toStr(tag)),
            tagLength: 128
        });
        decipher.update(forge.util.createBuffer(toStr(data)));
        var pass = decipher.finish();

        if (!pass) {
            throw new Error('Decryption failed: wrong password or corrupted data');
        }

        var innerCiphertext = fromStr(decipher.output.getBytes());
        // Inner decryption with random key
        return decrypt(innerCiphertext, key);
    }

    // --- Public API ---
    return {
        generateKey,
        exportKeyToBase64url,
        importKeyFromBase64url,
        encrypt,
        decrypt,
        deriveKeyFromPassword,
        encryptWithPassword,
        decryptWithPassword,
        stringToBytes,
        bytesToString,
        base64urlEncode,
        base64urlDecode,
        base64Encode,
        base64Decode
    };
})();
